/*
  32bitアイテムへのアクセス

  ターゲット:PRO, PRO+, X, MX

  モデルを間違えてコントロールテーブルにアクセスすると危険なため、
  対象のモデルが何であるかを判別してから処理を行う。
*/

#include  <stdio.h>
#include  <conio.h>
#include  "dx2lib.h"
#include  "dx2memmap.h"

// マクロ定義
#define _COMPORT  (5)       // デフォルトポート番号
#define _BAUDRATE (57600)   // デフォルトボーレート[bps]
#define _TARGETID (1)       // デフォルト対象ID

//------------------------------------------------
// 実行時にポート番号・ボーレート・IDを選択
//------------------------------------------------
bool InputNum (int32_t *num) {
  char buf[100];
  fgets(buf, sizeof(buf), stdin);
  rewind(stdin);
  return (strlen(buf) && sscanf(buf, "%d", num) == 1);
}

bool EditComAndBaudAndTarget (char *comport, uint32_t *comn, uint32_t *baud, uint8_t *id) {
  bool result = true;

  int32_t n;

  printf ("Input comport num (default %d) = ", _COMPORT);
  if (!InputNum (&n)) { n = _COMPORT; result = false; }
  if (n >= 1 && n <= 255) *comn = n; else *comn = _COMPORT;
  sprintf(comport, "\\\\.\\COM%d", *comn);

  printf ("Input baudrate (default %dbps) = ", _BAUDRATE);
  if (!InputNum (&n)) { n = _BAUDRATE; result = false; }
  if (n >= 9600 && n <= 3000000) *baud = n; else *baud = _BAUDRATE;

  printf ("Input target id (default %d) = ", _TARGETID);
  if (!InputNum (&n)) { n = _TARGETID; result = false; }
  if (n >= 1 && n < 0xfd) *id = n; else *id = _TARGETID;

  return result;
}

TDXL_DevType CheckModelInfo (TDeviceID dev, uint8_t id) {
  uint16_t modelno;
  if (DX2_ReadWordData (dev, id, 0, &modelno, NULL)) {
    switch (modelno) {
      case 0x015E:  //XL-320
        return devtXL320;
      case 0x001E:  //MX-28(2.0)
      case 0x0137:  //MX-64(2.0)
      case 0x0141:  //MX-106(2.0)
      case 0x0424:  //XL430-W250
      case 0x0406:  //XM430-W210
      case 0x03F2:  //XH430-W210
      case 0x041A:  //XH430-V210
      case 0x03FC:  //XM430-W350
      case 0x03E8:  //XH430-W350
      case 0x0410:  //XH430-V350
      case 0x046A:  //XM540-W150
      case 0x0456:  //XH540-W150
      case 0x047E:  //XH540-V150
      case 0x0460:  //XM540-W270
      case 0x044C:  //XH540-W270
      case 0x0474:  //XH540-V270
        return devtX;
      case 0xD308:  //H54-200-S500-R
      case 0xD208:  //H54-100-S500-R
      case 0xC800:  //H42-20-S300-R
      case 0xB510:  //M54-60-S250-R
      case 0xB410:  //M54-40-S250-R
      case 0xA918:  //M42-10-S260-R
      case 0x9520:  //L54-50-S290-R
      case 0x9508:  //L54-50-S500-R
      case 0x9428:  //L54-30-S400-R
      case 0x9408:  //L54-30-S500-R
      case 0x8900:  //L42-10-S300-R
        return devtPRO;
      case 0xD309:  //H54-200-S500-RA
      case 0xD209:  //H54-100-S500-RA
      case 0xC801:  //H42-20-S300-RA
      case 0xB511:  //M54-60-S250-RA
      case 0xB411:  //M54-40-S250-RA
      case 0xA919:  //M42-10-S260-RA
      case 0x07E4:  //H54P-200-S500-R
      case 0x07DA:  //H54P-100-S500-R
      case 0x07D0:  //H42P-020-S300-R
        return devtPROP;
      // その他
      default:
        return devtNONE;
    }
  }
  return devtNONE;
}

//------------------------------------------------
// メイン
//------------------------------------------------
void main (void) {
  uint16_t    adr_max, adr_min, adr_ppos, adr_pos, adr_ten; // 各アイテムのアドレス
  int32_t     maxpos, minpos, diffpos, pos;
  TDeviceID   dev;
  TErrorCode  err;

  char comname[20];
  uint32_t comnum, baud;
  uint8_t id;

  EditComAndBaudAndTarget (comname, &comnum, &baud, &id);
  printf("\nCOM=%s, Baudrate=%d, id=%d\n", comname, baud, id);

  // ポートオープン
  if ((dev = DX2_OpenPort (comname, baud))) {
    printf ("Open success\n");

    // モデル毎に各アイテムの対象アドレスを変更
    switch (CheckModelInfo (dev, id)) {
      case devtX:
        adr_max  = ADDRESS_X_MAX_POSITION_LIMIT;
        adr_min  = ADDRESS_X_MIN_POSITION_LIMIT;
        adr_ppos = ADDRESS_X_PRESENT_POSITION;
        adr_ten  = ADDRESS_X_TORQUE_ENABLE;
        adr_pos  = ADDRESS_X_GOAL_POSITION;
        break;
      case devtPRO:
        adr_max  = ADDRESS_PRO_MAX_POSITION_LIMIT;
        adr_min  = ADDRESS_PRO_MIN_POSITION_LIMIT;
        adr_ppos = ADDRESS_PRO_PRESENT_POSITION;
        adr_ten  = ADDRESS_PRO_TORQUE_ENABLE;
        adr_pos  = ADDRESS_PRO_GOAL_POSITION;
        break;
      case devtPROP:
        adr_max  = ADDRESS_PROP_MAX_POSITION_LIMIT;
        adr_min  = ADDRESS_PROP_MIN_POSITION_LIMIT;
        adr_ppos = ADDRESS_PROP_PRESENT_POSITION;
        adr_ten  = ADDRESS_PROP_TORQUE_ENABLE;
        adr_pos  = ADDRESS_PROP_GOAL_POSITION;
        break;
      default:
        adr_max  = 0;
        adr_min  = 0;
        adr_ppos = 0;
        adr_ten  = 0;
        adr_pos  = 0;
        break;
    }
    if (adr_max != 0) {
      // 初期条件の種痘とトルクイネーブル
      if (
        // 最大角度取得
        DX2_ReadLongData (dev, id, adr_max, (uint32_t *)&maxpos, &err) &&
        // 最小角度取得
        DX2_ReadLongData (dev, id, adr_min, (uint32_t *)&minpos, &err) &&
        // 現在角度取得
        DX2_ReadLongData (dev, id, adr_ppos, (uint32_t *)&pos, &err) &&
        // トルクイネーブル
        DX2_WriteByteData (dev, id, adr_ten, 1, &err)
      ) {
        DX2_WriteByteData (dev, id, adr_ten, 1, &err);
        diffpos = (maxpos - minpos) / 1000;
        printf ("maxpos=%d, minpos=%d, presentpos=%d\n", maxpos, minpos, pos);

        // 何かキーを押すとループを抜ける
        while (!kbhit()) {
          pos += diffpos;
          if (pos >= maxpos) {
            pos = maxpos;
            diffpos *= -1;
          } else if (pos <= minpos) {
            pos = minpos;
            diffpos *= -1;
          }
          // 目標角度指令
          if (DX2_WriteLongData (dev, id, adr_pos, (uint32_t)pos, &err))
            printf ("Sent position = %7d [%04X]\r\n", pos, err);
          else
            printf ("Sent position error [$%04X]\n", err);
          // 1ミリ秒待ち
          Sleep (1);
        }
        // トルクディスエーブル
        DX2_WriteByteData (dev, id, adr_ten, 0, &err);
        printf ("\n");
      }
    } else printf ("Not supported device ID=%d\n", id);

    // ポートクローズ
    DX2_ClosePort (dev);
  } else printf ("Open error\n");

  printf ("Fin\n");
}
